#ifndef __CPresetIO__
#define __CPresetIO__

//	===========================================================================

#include "../Collections/CollectionTypedef.h"
#include "CTextStream.hpp"

//	===========================================================================

using Exponent::Collections::TStringCountedPointerArray;
using Exponent::IO::CTextStream;

//	===========================================================================

namespace Exponent
{
	namespace IO
    {
		/**
		 * @deprecated This class is here for historical use.
		 * @see CXMLDocument
		 * @see CTextStream
		 * @see CFileStream
		 * @class CPresetIO CPresetIO.hpp
		 * @brief Textual preset saving to a stream
		 * @deprecated
		 * @see CXMLDocument
		 *
		 * @date 28/08/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CPresetIO.hpp,v 1.5 2007/02/08 21:06:44 paul Exp $
		 */
        class CPresetIO : public CCountedObject
        {
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

            /**
			 * Construction
			 */
            CPresetIO();

            /**
			 * Destruction
			 */
            virtual ~CPresetIO();

//	===========================================================================

			/**
			 * Open for writing
			 * @param streamMode The mode of io
			 * @param pluginName A descriptive string that is used to identify the type of preset format
			 * @param presetPath The path to the preset to read / write
			 * @retval bool True if opened properly, false otherwise
			 */
			bool open(const CTextStream::EStreamMode streamMode, const CString &pluginName, const CSystemString &presetPath);

			/**
			 * Close the file
			 * @param pluginName A descriptive string that is used to identify the type of preset format
			 * @retval bool True if closed properly, false otherwise
			 */
			bool close(const CString &pluginName);

			/**
			 * Close the file (immediate close, no checking)
			 */
			void close();

//	===========================================================================

			/**
			 * Read the version number
			 * @param expectedVersion The version number you expect to find
			 * @retval bool True if versions match, false otherwise
			 */
			bool readVersionNumber(const long expectedVersion);

			/**
			 * Read the version number
			 * @param expectedVersion The version number you expect to find
			 * @param numberOfVersions The size of expected versions
			 * @param actualVersion On return is filled with the version number or undefined on error
			 * @retval bool True if versions match, false otherwise
			 */
			bool readVersionNumber(const long *expectedVersion, const long numberOfVersions, long &actualVersion);

			/**
			 * Read head chunk
			 * @param array Array to store strings contained within the head chunk
			 * @retval bool True if read correctly, false on error
			 */
			bool readHeadChunk(TStringCountedPointerArray &array);

			/**
			 * Open the head chunk
			 * @retval bool True if read opening head flag correctly, false otherwise
			 */
			bool openHeadChunk();

			/**
			 * Open the body chunk
			 * @retval bool True if read opening body flag correctly, false otherwise
			 */
			bool openBodyChunk();

			/**
			 * Read the parameters as a chunk of doubles
			 * @param numberOfParameters The size of the array to read in
			 * @param parameters On return is filled with the parameters as read from disk
			 * @retval bool True if parameters read correctly
			 */
			bool readParameterChunk(const long numberOfParameters, double *parameters);

			/**
			 * Read a chunk to doubles
			 * @param tag A descriptive chunk tag expected
			 * @param sizeOfChunk The size of the chunk to write in elements
			 * @param buffer The buffer to fill
			 * @retval bool True if read correctly, false otherwise
			 */
			bool readChunkToDoubles(const CString &tag, const long sizeOfChunk, double *buffer);

			/**
			 * Read a chunk to strings
			 * @param tag A descriptive chunk tag expected
			 * @param array Array to store strings contained within the chunk
			 * @retval bool True if read the chunk properly, false otherwise
			 */
			bool readChunkToStrings(const CString &tag, TStringCountedPointerArray &array);

			/**
			 * Read a chunk to longs
			 * @param tag A descriptive chunk tag expected
			 * @param sizeOfChunk The size of the chunk to write in elements
			 * @param buffer The buffer to fill
			 * @retval bool True if read correctly, false otherwise
			 */
			bool readChunkToLongs(const CString &tag, const long sizeOfChunk, long *buffer);

//	===========================================================================

			/**
			 * Write the version number
			 * @param number The version number
			 * @retval bool True if read correctly, false otherwise
			 */
			bool writeVersionNumber(const long number);

			/**
			 * Write head chunk
			 * @param array The array of strings to write to the head chunk
			 * @retval bool True if written correctly, false otherwise
			 */
			bool writeHeadChunk(TStringCountedPointerArray &array);

			/**
			 * Write the parameters
			 * @param numberOfParameters The number of parameters to write
			 * @param parameters The parameters to write
			 * @retval bool True if written correctly, false otherwise
			 */
			bool writeParameterChunk(const long numberOfParameters, const double *parameters);

			/**
			 * Write chunk to doubles
			 * @param tag A descriptive chunk tag
			 * @param sizeOfChunk The number of elements to write
			 * @param buffer The buffer to write
			 * @retval bool True if written correctly, false otherwise
			 */
			bool writeChunkToDoubles(const CString &tag, const long sizeOfChunk, const double *buffer);

			/**
			 * Write chunk to strings
			 * @param tag A descriptive chunk tag
			 * @param array The array of strings to write
			 * @retval bool True if written correctly, false otherwise
			 */
			bool writeChunkToStrings(const CString &tag, TStringCountedPointerArray &array);

			/**
			 * Write chunk to longs
			 * @param tag A descriptive chunk tag
			 * @param sizeOfChunk The number of elements to write
			 * @param buffer The buffer to write
			 * @retval bool True if written correctly, false otherwise
			 */
			bool writeChunkToLongs(const CString &tag, const long sizeOfChunk, long *buffer);

//	===========================================================================

			/**
			 * Close the head chunk
			 * @retval bool True if head tag closed, false otherwise
			 */
			bool closeHeadChunk();

			/**
			 * Close the body chunk
			 * @retval bool True if body tag closed, false otherwise
			 */
			bool closeBodyChunk();

//	===========================================================================

			/**
			 * Get the stream
			 * @retval CTextStream & The text stream
			 */
			CTextStream &getTextStream() { return m_stream; }

//	===========================================================================

        protected:

//	===========================================================================

			/**
			 * Write an opening tag - adds < before the tag and > at the end of the tag
			 * @param tag The tag to write
			 * @retval bool True if written properly, false otherwise
			 */
			bool writeOpenTag(const CString &tag);

			/**
			 * Write a close tag - adds </ before the tag and > at the end of the tag
			 * @retval bool True if written properly, false otherwise
			 */
			bool writeCloseTag(const CString &tag);

//	===========================================================================

			/**
			 * Read an opening tag
			 * @param tag The tag to read (doesnt need to be surrounded by \< \>)
			 * @retval bool True if read properly, false otherwise
			 */
			bool readOpenTag(const CString &tag);

			/**
			 * Read a close tag
			 * @param tag The tag to read (doesnt need to be surrounded by \<\/ \>)
			 * @retval bool True if read properly, false otherwise
			 */
			bool readCloseTag(const CString &tag);

//	===========================================================================

			/**
			 * Get an open tag - converts 'text' to '\<\/text\>'
			 * @param openTag On return filled with the tag
			 * @param tag The tag to make in to an open tag
			 */
			void constructOpenTag(CString &openTag, const CString &tag);

			/**
			 * Get a close tag - converts 'text' to '\<\/text\>'
			 * @param closeTag On return filled with the tag
			 * @param tag The tag to make in to a closing tag
			 */
			void constructCloseTag(CString &closeTag, const CString &tag);

//	===========================================================================

			CTextStream m_stream;				/**< The actual output stream */

			bool m_isOpenForRead;				/**< Can we read */
			bool m_isOpenForWrite;				/**< Can we write */


        };
    }
}
#endif    // End of CPresetIO.hpp
